<?php
/**
 * Reformer for Elementor
 * Customizable contact form for Elementor editor
 * Exclusively on https://1.envato.market/reformer-elementor
 *
 * @encoding        UTF-8
 * @version         1.0.2
 * @copyright       (C) 2018 - 2021 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmitry Merkulov (dmitry@merkulov.design), Cherviakov Vlad (vladchervjakov@gmail.com)
 * @support         help@merkulov.design
 **/

namespace Merkulove\ReformerElementor;

use Merkulove\ReformerElementor\Unity\AdminScripts;
use Merkulove\ReformerElementor\Unity\AdminStyles;
use Merkulove\ReformerElementor\Unity\Plugin;
use Merkulove\ReformerElementor\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

class AddressBookMetaBoxOptions
{
    /**
     * The one true AddressBookMetaBoxOptions.
     *
     * @var AddressBookMetaBoxOptions
     * @since 1.0.0
     **/
    private static $instance;


    /**
     * Sets up a new instance.
     *
     * @since 1.0.0
     * @access public
     **/
    private function __construct() {

    }

    /**
     * Render "Options" metabox with all fields.
     *
     * @param $reformer_form
     * @since 1.0.0
     */
    public function render_meta_box( $reformer_form ) {

        /** Render Nonce field to validate on save. */
        $this->render_nonce();

        ?>
        <div class="mdp-options-box">
            <table class="form-table">
                <tbody>
                <?php

                /** Render Contact Email */
                $this->contact_email( $reformer_form );

                /** Render Contact Discord Webhook */
                $this->contact_discord_webhook( $reformer_form );

                /** Render Contact Slack Webhook */
                $this->contact_slack_webhook( $reformer_form );

                ?>
                </tbody>
            </table>
        </div>
        <?php
    }


    /**
     * Render Contact Email field.
     *
     * @param $reformer_form
     * @since 1.0.0
     */
    private function contact_email( $reformer_form ) {
        ?>
        <tr>
            <th scope="row">
                <label for="mdp_form_contact_name"><?php esc_html_e( 'Email:', 'reformer-elementor' ); ?></label>
            </th>
            <td>
                <div class="mdp-control-field">
                <?php
                    /** Get value from meta if it's already been entered. */
                    $value = get_post_meta( $reformer_form->ID, 'mdp_field_contact_email', true );

                    UI::get_instance()->render_input(
                        $value,
                        esc_html__( 'Contact email', 'reformer-elementor' ),
                        esc_html__( 'Enter the email of the contact', 'reformer-elementor' ),
                        [
                            'name' => 'mdp_field_contact_email',
                            'id' => 'mdp_field_contact_email',
                        ]
                    );
                ?>
                </div>
            </td>
        </tr>
        <?php
    }

    /**
     * Render Contact Discord Webhook field.
     *
     * @param $reformer_form
     * @since 1.0.0
     */
    private function contact_discord_webhook( $reformer_form ) {
        ?>
        <tr>
            <th scope="row">
                <label for="mdp_form_contact_name"><?php esc_html_e( 'Discord webhook:', 'reformer-elementor' ); ?></label>
            </th>
            <td>
                <div class="mdp-control-field">
                <?php
                    /** Get value from meta if it's already been entered. */
                    $value = get_post_meta( $reformer_form->ID, 'mdp_field_discord_webhook', true );

                    UI::get_instance()->render_input(
                        $value,
                        esc_html__( 'Discord webhook', 'reformer-elementor' ),
                        wp_kses_post( 'Enter discord webhook url. 
                                    Visit <a href="https://support.discord.com/hc/en-us/articles/228383668-Intro-to-Webhooks" 
                                    target="_blank">Online Documentation</a> for more details.' ),
                        [
                            'name' => 'mdp_field_discord_webhook',
                            'id' => 'mdp_field_discord_webhook',
                        ]
                    );
                ?>
                </div>
            </td>
        </tr>
        <?php
    }

    /**
     * Render Contact Slack Webhook field.
     *
     * @param $reformer_form
     * @since 1.0.0
     */
    private function contact_slack_webhook( $reformer_form ) {
        ?>
        <tr>
            <th scope="row">
                <label for="mdp_form_contact_name"><?php esc_html_e( 'Slack webhook:', 'reformer-elementor' ); ?></label>
            </th>
            <td>
                <div class="mdp-control-field">
                <?php
                    /** Get value from meta if it's already been entered. */
                    $value = get_post_meta( $reformer_form->ID, 'mdp_field_slack_webhook', true );

                    UI::get_instance()->render_input(
                        $value,
                        esc_html__( 'Slack webhook', 'reformer-elementor' ),
                        wp_kses_post( 'Enter slack webhook url. 
                                    Visit <a href="https://api.slack.com/messaging/webhooks#getting_started" 
                                    target="_blank">Online Documentation</a> for more details.' ),
                        [
                            'name' => 'mdp_field_slack_webhook',
                            'id' => 'mdp_field_slack_webhook',
                        ]
                    );
                ?>
                </div>
            </td>
        </tr>
        <?php
    }

    /**
     * Saves metabox options.
     *
     * @param $post_id
     * @since 1.0.0
     */
    public function save_metabox( $post_id ) {
        /** Options fields keys. */
        $key = [
            'mdp_field_contact_email', // Contact Email.
            'mdp_field_discord_webhook', // Contact Discord Webhook.
            'mdp_field_slack_webhook', // Contact Slack Webhook.
        ];

        /** Save each field. */
        foreach ( $key as $field ) {
            $value = ( isset( $_POST[$field] ) ? wp_kses_post( $_POST[$field] ) : '' );

            update_post_meta( $post_id, $field, $value );
        }
    }




    /**
     * Render Nonce field to validate form request came from current site.
     *
     * @since 1.0.0
     **/
    private function render_nonce() {

        wp_nonce_field( Plugin::get_basename(), 'options_metabox_fields_nonce' );

    }


    /**
     * Main AddressBookMetaBoxOptions Instance.
     *
     * Insures that only one instance of AddressBookMetaBoxOptions exists in memory at any one time.
     *
     * @static
     * @return AddressBookMetaBoxOptions
     * @since 1.0.0
     **/
    public static function get_instance() {
        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof AddressBookMetaBoxOptions ) ) {
            self::$instance = new AddressBookMetaBoxOptions();
        }

        return self::$instance;
    }

}