<?php
/**
 * Reformer for Elementor
 * Customizable contact form for Elementor editor
 * Exclusively on https://1.envato.market/reformer-elementor
 *
 * @encoding        UTF-8
 * @version         1.0.2
 * @copyright       (C) 2018 - 2021 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmitry Merkulov (dmitry@merkulov.design), Cherviakov Vlad (vladchervjakov@gmail.com)
 * @support         help@merkulov.design
 **/

namespace Merkulove\ReformerElementor;

use Merkulove\ReformerElementor\Unity\Plugin;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

/**
 * SINGLETON: The class used to implement admin scripts enqueueing.
 *
 * @since 1.0.0
 *
 **/
final class AdminScripts {

    /**
     * The one true AdminScripts.
     *
     * @var AdminScripts
     * @since 1.0.0
     **/
    private static $instance;

    /**
     * Sets up a new AdminScripts instance.
     *
     * @since 1.0.0
     * @access public
     **/
    private function __construct() {

        /** Add admin javascript. */
        add_action( 'admin_enqueue_scripts', [ $this, 'admin_scripts' ] );

    }

    /**
     * Add JS for admin area.
     *
     * @since   1.0.0
     * @return void
     **/
    public function admin_scripts() {

        /** ReFormer Settings Page. */
        $this->settings_scripts();

        /** Edit ReFormer page. */
        $this->edit_scripts();

    }

    /**
     * Scripts for edit ReFormer page.
     *
     * @since   1.0.0
     * @return void
     **/
    private function edit_scripts() {

        /** Work only on Edit ReFormer page. */
        $screen = get_current_screen();
        if ( null === $screen ) { return; }
        if ( 'mdp_address_book_cpt' !== $screen->id ) { return; }

        /** Add class .mdc-disable to body. So we can use UI without overrides WP CSS, only for this page. */
        add_action( 'admin_body_class', [$this, 'add_admin_class'] );

        wp_enqueue_script(
            'merkulov-ui',
               Plugin::get_url() . 'src/Merkulove/Unity/assets/js/merkulov-ui' . Plugin::get_suffix() . '.js',
                   [], Plugin::get_version(),
           true
        );



    }

    /**
     * Add class to body in admin area.
     *
     * @param string $classes - Space-separated list of CSS classes.
     *
     * @since 1.0.0
     * @return string
     */
    public function add_admin_class( $classes ) {

        return $classes . ' mdc-disable ';

    }


    /**
     * Scripts for Plugin Settings page.
     *
     * @since   1.0.0
     * @return void
     **/
    private function settings_scripts() {

        /** Work only on Plugin Settings Page. */
        $screen = get_current_screen();
        if ( null === $screen ) { return; }
        if ( $screen->base !== 'mdp_reformer_sub_page_mdp_reformer_elementor_settings' ) { return; }

        wp_enqueue_script(
            'merkulov-ui',
               Plugin::get_url() . 'src/Merkulove/Unity/assets/js/merkulov-ui' . Plugin::get_suffix() . '.js',
                   [], Plugin::get_version(),
          true
        );

    }

    /**
     * Main AdminScripts Instance.
     *
     * Insures that only one instance of AdminScripts exists in memory at any one time.
     *
     * @static
     * @return AdminScripts
     * @since 1.0.0
     **/
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

            self::$instance = new self;

        }

        return self::$instance;

    }

} // End Class AdminScripts.
