<?php /** @noinspection PhpUndefinedClassInspection */
/**
 * Reformer for Elementor
 * Customizable contact form for Elementor editor
 * Exclusively on https://1.envato.market/reformer-elementor
 *
 * @encoding        UTF-8
 * @version         1.0.2
 * @copyright       (C) 2018 - 2021 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmitry Merkulov (dmitry@merkulov.design), Cherviakov Vlad (vladchervjakov@gmail.com)
 * @support         help@merkulov.design
 **/

namespace Merkulove\ReformerElementor;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

use Exception;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Core\Schemes\Typography;
use Elementor\Core\Schemes\Color;
use Elementor\Utils;
use Elementor\Repeater;
use Merkulove\ReformerElementor\Unity\Plugin as UnityPlugin;
use Merkulove\ReformerElementor\Unity\Settings;

/** @noinspection PhpUnused */
/**
 * Reformer - Custom Elementor Widget.
 **/
class reformer_elementor extends Widget_Base {

    /**
     * Use this to sort widgets.
     * A smaller value means earlier initialization of the widget.
     * Can take negative values.
     * Default widgets and widgets from 3rd party developers have 0 $mdp_order
     **/
    public $mdp_order = 1;

    /**
     * Widget base constructor.
     * Initializing the widget base class.
     *
     * @access public
     * @throws Exception If arguments are missing when initializing a full widget instance.
     * @param array      $data Widget data. Default is an empty array.
     * @param array|null $args Optional. Widget default arguments. Default is null.
     *
     * @return void
     **/
    public function __construct( $data = [], $args = null ) {

        parent::__construct( $data, $args );

        wp_register_style(
        'mdp-reformer-elementor-admin',
        UnityPlugin::get_url() . 'src/Merkulove/Unity/assets/css/elementor-admin' . UnityPlugin::get_suffix() . '.css',
            [],
            UnityPlugin::get_version()
        );
        wp_register_style(
        'mdp-reformer-elementor',
        UnityPlugin::get_url() . 'css/reformer-elementor' . UnityPlugin::get_suffix() . '.css',
            [],
            UnityPlugin::get_version()
        );
	    wp_register_script(
        'mdp-reformer-elementor',
        UnityPlugin::get_url() . 'js/reformer-elementor' . UnityPlugin::get_suffix() . '.js',
            [ 'jquery', 'elementor-frontend' ],
            UnityPlugin::get_version(),
        true
        );

        /** Pass variables to script */
        wp_localize_script( 'mdp-reformer-elementor', 'mdpReformerElementorWP', [
            'endpoint' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'reformer-elementor-nonce' )
        ] );

    }

    /**
     * Return a widget name.
     *
     * @return string
     **/
    public function get_name() {

        return 'mdp-reformer-elementor';

    }

    /**
     * Return the widget title that will be displayed as the widget label.
     *
     * @return string
     **/
    public function get_title() {

        return esc_html__( 'ReFormer', 'reformer-elementor' );

    }

    /**
     * Set the widget icon.
     *
     * @return string
     */
    public function get_icon() {

        return 'mdp-reformer-elementor-widget-icon';

    }

    /**
     * Set the category of the widget.
     *
     * @return array with category names
     **/
    public function get_categories() {

        return [ 'general' ];

    }

    /**
     * Get widget keywords. Retrieve the list of keywords the widget belongs to.
     *
     * @access public
     *
     * @return array Widget keywords.
     **/
    public function get_keywords() {

        return [ 'Merkulove', 'Reformer' ];

    }

    /**
     * Get style dependencies.
     * Retrieve the list of style dependencies the widget requires.
     *
     * @access public
     *
     * @return array Widget styles dependencies.
     **/
    public function get_style_depends() {

        return [ 'mdp-reformer-elementor', 'mdp-reformer-elementor-admin', 'elementor-icons-fa-solid' ];

    }

	/**
	 * Get script dependencies.
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @access public
     *
	 * @return array Element scripts dependencies.
	 **/
	public function get_script_depends() {

		return [ 'mdp-reformer-elementor' ];

    }

    /**
     * Add the widget controls.
     *
     * @access protected
     * @return void with category names
     **/
    protected function register_controls() {

        /** Content Tab. */
        $this->tab_content();

        /** Style Tab. */
        $this->tab_style();

    }

    /**
     * Add widget controls on Content tab.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function tab_content() {

        /** Content -> General Content Section. */
        $this->section_content_general();

        /** Content -> Messages Content Section. */
        $this->section_content_messages();

        /** Content -> Form Fields Content Section. */
        $this->section_content_form_fields();

        /** Content -> Form Email Content Section. */
        $this->section_content_email();

        /** Content -> Discord Content Section. */
        $this->section_content_discord();

        /** Content -> Discord Content Section. */
        $this->section_content_slack();

        /** Content -> Telegram Content Section */
        $this->section_content_telegram();

    }

    /**
     * Add widget controls on Style tab.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function tab_style() {

        /** Style -> Section Style Title. */
        $this->section_style_title();

        /** Style -> Section Style Description. */
        $this->section_style_description();

        /** Style -> Section Style Label. */
        $this->section_style_label();

        /** Style -> Section Style Input. */
        $this->section_style_input();

        /** Style -> Section Style Range. */
        $this->section_style_range();

        /** Style -> Section Style Textarea. */
        $this->section_style_textarea();

        /** Style -> Section Style Radio Button. */
        $this->section_style_radio_button();

        /** Style -> Section Style Checkbox. */
        $this->section_style_checkbox();

        /** Style -> Section Style Select */
        $this->section_style_select();

        /** Style -> Section Style Button */
        $this->section_style_button();

        /** Style -> Section Style Errors */
        $this->section_style_success();

        /** Style -> Section Style Errors */
        $this->section_style_errors();

    }


    /**
     * Get all contacts from custom book.
     *
     * @return int[]|\WP_Post[]
     * @since 1.0.0
     * @access private
     *
     */
    private function get_address_book_contacts() {
        return get_posts(
            [
                'post_type' => 'mdp_address_book_cpt',
                'post_status' => 'publish',
                'ignore_sticky_posts' => 1,
                'numberposts' => -1
            ]
        );
    }

    /**
     * Add widget controls: Content -> General Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_general() {

        $this->start_controls_section( 'section_content_general', [
            'label' => esc_html__( 'General', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_CONTENT
        ] );

        $this->add_control(
            'form_name',
            [
                'label' => esc_html__( 'Form name', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type form name here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_responsive_control(
            'form_align',
            [
                'label' => esc_html__( 'Fields alignment', 'reformer-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => esc_html__( 'Left', 'reformer-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'reformer-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'flex-end' => [
                        'title' => esc_html__( 'Right', 'reformer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                    'space-between' => [
                        'title' => esc_html__( 'Space between', 'reformer-elementor' ),
                        'icon'  => 'fa fa-expand-arrows-alt',
                    ],
                    'space-around'  => [
                        'title' => esc_html__( 'Space around', 'reformer-elementor' ),
                        'icon'  => 'fa fa-compress-arrows-alt',
                    ],
                ],
                'selectors' => [
                        '{{WRAPPER}} .mdp-reformer-elementor-form' => 'justify-content: {{VALUE}}'
                ],
                'default' => 'flex-start',
                'toggle' => true,
            ]
        );

        $this->add_responsive_control(
            'button_align',
            [
                'label' => esc_html__( 'Submit button alignment', 'reformer-elementor' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => esc_html__( 'Left', 'reformer-elementor' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'reformer-elementor' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'flex-end' => [
                        'title' => esc_html__( 'Right', 'reformer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-reformer-elementor-submit-button-wrapper' => 'justify-content: {{VALUE}}'
                ],
                'default' => 'center',
                'toggle' => true,
            ]
        );

        $this->add_control(
            'show_title',
            [
                'label' => esc_html__( 'Show title', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'form_title',
            [
                'label' => esc_html__( 'Title', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type title here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                    'show_title' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'show_description',
            [
                'label' => esc_html__( 'Show description', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'form_description',
            [
                'label' => esc_html__( 'Description', 'reformer-elementor' ),
                'type' => Controls_Manager::WYSIWYG,
                'placeholder' => esc_html__( 'Type your description here', 'reformer-elementor' ),
                'condition' => [
                    'show_description' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'submit_button_text',
            [
                'label' => esc_html__( 'Submit button text', 'reformer-elementor' ),
                'label_block' => true,
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Submit', 'reformer-elementor' ),
                'placeholder' => esc_html__( 'Type button text here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'send_referrer',
            [
                'label' => esc_html__( 'Current post url in submissions', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'send_to',
            [
                'label' => esc_html__( 'Send to', 'reformer-elementor' ),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => [
                    'save' => esc_html__( 'Save', 'reformer-elementor' ),
                    'telegram' => esc_html__( 'Telegram', 'reformer-elementor' ),
                    'email' => esc_html__( 'Email', 'reformer-elementor' ),
                    'discord' => esc_html__( 'Discord', 'reformer-elementor' ),
                    'slack' => esc_html__( 'Slack', 'reformer-elementor' ),
                ],
                'default' => [ 'save', 'email' ],
            ]
        );

        $this->add_control(
            'record_suffix',
            [
                'label' => esc_html__( 'Record suffix', 'reformer-elementor' ),
                'label_block' => true,
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type field name here', 'reformer-elementor' ),
                'description' => esc_html__( 'Enter the name of the field, the content of which will be added 
                                                as a suffix to the name of the record', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->end_controls_section();

    }

    /**
     * Add widget controls: Content -> Messages Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_messages() {
        $this->start_controls_section( 'section_content_messages', [
            'label' => esc_html__( 'Messages', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_CONTENT
        ] );

        $this->add_control(
            'success_message',
            [
                'label' => esc_html__( 'Success message', 'reformer-elementor' ),
                'label_block' => true,
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type success message here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'success_message_text_align',
            [
                'label' => esc_html__( 'Success message text align', 'reformer-domain' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'reformer-domain' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'reformer-domain' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'reformer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-reformer-elementor-success-message' => 'text-align: {{VALUE}}'
                ],
                'default' => 'center',
                'toggle' => true,
            ]
        );

        $this->add_control(
            'error_message',
            [
                'label' => esc_html__( 'Error message', 'reformer-elementor' ),
                'label_block' => true,
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type error message here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'error_message_text_align',
            [
                'label' => esc_html__( 'Error message text align', 'reformer-domain' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'reformer-domain' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'reformer-domain' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'reformer-elementor' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-reformer-elementor-error-message' => 'text-align: {{VALUE}}'
                ],
                'default' => 'center',
                'toggle' => true,
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Add widget controls: Content -> Email Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_email() {

        $this->start_controls_section( 'section_content_email', [
            'label' => esc_html__( 'Email', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_CONTENT,
            'condition' => [ 'send_to' => 'email' ]
        ] );

        $this->add_control(
            'use_address_book_email',
            [
                'label' => esc_html__( 'Use address book', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'address_book_email',
            [
                'label' => esc_html__( 'Contact', 'reformer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'text',
                'options' => wp_list_pluck( $this->get_address_book_contacts(), 'post_title', 'ID' ),
                'condition' => [ 'use_address_book_email' => 'yes' ]
            ]
        );


        $this->add_control(
            'email_to',
            [
                'label' => esc_html__( 'To', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => esc_html__( get_option( 'admin_email' ), 'reformer-elementor' ),
                'placeholder' => esc_html__( 'Type your email here', 'reformer-elementor' ),
                'condition' => [ 'use_address_book_email!' => 'yes' ]
            ]
        );

        $this->add_control(
            'email_subject',
            [
                'label' => esc_html__( 'Subject', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => esc_html__( 'Message from '.get_bloginfo( 'name' ), 'reformer-elementor' ),
                'placeholder' => esc_html__( 'Type your email here', 'reformer-elementor' ),
            ]
        );

        $this->add_control(
            'email_message',
            [
                'label' => esc_html__( 'Message', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 10,
                'default' => esc_html__( '[form-fields]', 'reformer-elementor' ),
                'placeholder' => esc_html__( 'Type your message here', 'reformer-elementor' ),
                'description' => 'Place [form-fields] for form fields data.',
            ]
        );

        $this->add_control(
            'email_from',
            [
                'label' => esc_html__( 'Email from', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => esc_html__( 'email_from@localhost', 'reformer-elementor' ),
                'placeholder' => esc_html__( 'Type your email from', 'reformer-elementor' ),
            ]
        );

        $this->add_control(
            'email_from_name',
            [
                'label' => esc_html__( 'From name', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => esc_html__( get_bloginfo( 'name' ), 'reformer-elementor' ),
                'placeholder' => esc_html__( 'Type your name here', 'reformer-elementor' ),
            ]
        );

        $this->end_controls_section();

    }


    /**
     * Add widget controls: Content -> Discord Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_discord() {

        $this->start_controls_section( 'section_content_discord', [
            'label' => esc_html__( 'Discord', 'reformer-elementor' ),
            'tab' => Controls_Manager::TAB_CONTENT,
            'condition' => [ 'send_to' => 'discord' ]
        ] );

        $this->add_control(
            'use_address_book_discord',
            [
                'label' => esc_html__( 'Use address book', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'address_book_discord',
            [
                'label' => esc_html__( 'Contact', 'reformer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'text',
                'options' => wp_list_pluck( $this->get_address_book_contacts(), 'post_title', 'ID' ),
                'condition' => [ 'use_address_book_discord' => 'yes' ]
            ]
        );

        $this->add_control(
            'discord_webhook',
            [
                'label' => esc_html__( 'Discord webhook', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => 'https://discord.com/api/webhooks/',
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [ 'use_address_book_discord!' => 'yes' ],
                'description' => wp_kses_post( 'Enter discord webhook url. 
                                    Visit <a href="https://support.discord.com/hc/en-us/articles/228383668-Intro-to-Webhooks" 
                                    target="_blank">Online Documentation</a> for more details.' ),

            ]
        );

        $this->add_control(
            'discord_title',
            [
                'label' => esc_html__( 'Title', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type your title here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );


        $this->add_control(
            'discord_description',
            [
                'label' => esc_html__( 'Description', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type your description here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'discord_username',
            [
                'label' => esc_html__( 'Username', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type username here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'discord_avatar_url',
            [
                'label' => esc_html__( 'Avatar', 'reformer-elementor' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'discord_timestamp',
            [
                'label' => esc_html__( 'Timestamp', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'discord_form_fields',
            [
                'label' => esc_html__( 'Form fields data', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'frontend_available' => true,
            ]
        );

        $this->add_control(
            'discord_color',
            [
                'label' => esc_html__( 'Color', 'reformer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'alpha' => false,
                'default' => '#12C4F3',
            ]
        );

        $this->end_controls_section();
    }


    /**
     * Add widget controls: Content -> Slack Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_slack() {

        $this->start_controls_section( 'section_content_slack', [
            'label' => esc_html__( 'Slack', 'reformer-elementor' ),
            'tab' => Controls_Manager::TAB_CONTENT,
            'condition' => [ 'send_to' => 'slack' ]
        ] );

        $this->add_control(
            'use_address_book_slack',
            [
                'label' => esc_html__( 'Use address book', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'address_book_slack',
            [
                'label' => esc_html__( 'Contact', 'reformer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'text',
                'options' => wp_list_pluck( $this->get_address_book_contacts(), 'post_title', 'ID' ),
                'condition' => [ 'use_address_book_slack' => 'yes' ]
            ]
        );

        $this->add_control(
            'slack_webhook',
            [
                'label' => esc_html__( 'Slack webhook', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'placeholder' => 'https://hooks.slack.com/services/',
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [ 'use_address_book_slack!' => 'yes' ],
                'description' => wp_kses_post( 'Enter slack webhook url. 
                                    Visit <a href="https://api.slack.com/messaging/webhooks#getting_started" 
                                    target="_blank">Online Documentation</a> for more details.' ),
            ]
        );

        $this->add_control(
            'slack_title',
            [
                'label' => esc_html__( 'Title', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type your title here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );


        $this->add_control(
            'slack_description',
            [
                'label' => esc_html__( 'Description', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type your description here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'slack_username',
            [
                'label' => esc_html__( 'Username', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type username here', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'slack_timestamp',
            [
                'label' => esc_html__( 'Timestamp', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'slack_form_fields',
            [
                'label' => esc_html__( 'Form fields data', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'slack_color',
            [
                'label' => esc_html__( 'Color', 'reformer-elementor' ),
                'type' => Controls_Manager::COLOR,
                'alpha' => false,
                'default' => '#12C4F3',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Get telegram bot data.
     *
     * @return array|void
     * @since 1.0.0
     * @access private
     *
     */
    private function get_telegram_bot_data( $token ) {

        $url = 'https://api.telegram.org/bot'.$token.'/getupdates';

        if ( empty( $token ) ) { return;  }

        return wp_remote_get( $url );
    }

    /**
     * Add widget controls: Content -> Telegram Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_telegram() {

        $options = Settings::get_instance()->options;

        $token = $options['telegram_bot_token'];

        $this->start_controls_section('section_content_telegram', [
            'label' => esc_html__('Telegram', 'reformer-elementor'),
            'tab' => Controls_Manager::TAB_CONTENT,
            'condition' => ['send_to' => 'telegram']
        ]);

        if ( !empty( $token ) ) {

            $bot_data = json_decode( $this->get_telegram_bot_data( $token )['body'] );

            $options = [];

            add_option( 'mdp_telegram_chats', $options );

            if ( empty( $bot_data->result ) ) {
                // if api returns empty array get from db
                $options = get_option( 'mdp_telegram_chats' );
            } else {
                foreach ( $bot_data->result as $item ) {
                    if ( !isset( $item->message->chat ) ) {
                        continue;
                    }
                    $options[$item->message->chat->id] = $item->message->chat->title;
                }
                if ( $options !== get_option( 'mdp_telegram_chats' ) ) {
                    update_option( 'mdp_telegram_chats', $options );
                }
            }


            $this->add_control(
                'telegram_chat',
                [
                    'label' => esc_html__('Chat', 'reformer-elementor'),
                    'type' => Controls_Manager::SELECT,
                    'default' => !empty($options) ? array_keys($options)[0] : '',
                    'options' => $options
                ]
            );

            $this->add_control(
                'telegram_message',
                [
                    'label' => esc_html__('Message', 'reformer-elementor'),
                    'type' => Controls_Manager::TEXTAREA,
                    'rows' => 10,
                    'default' => esc_html__('[form-fields]', 'reformer-elementor'),
                    'placeholder' => esc_html__('Type your message here', 'reformer-elementor'),
                    'description' => 'Place [form-fields] for form fields data.',
                ]
            );

        } else {
            $this->add_control(
                'settings_link',
                [
                    'type' => Controls_Manager::RAW_HTML,
                    'raw' => '<strong>' . esc_html__( 'Telegram bot token not set.', 'reformer-elementor' ) .
                        '</strong><br>' . sprintf( wp_kses_post( 'Go to the <a href="%s" target="_blank">Settings</a> to set token.' ),
                            admin_url( 'admin.php?page=mdp_reformer_elementor_settings' ) ),
                    'separator' => 'after',
                    'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
                ]
            );
        }

        $this->end_controls_section();

    }

    /**
     * Add widget controls: Content -> General Content Section.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_content_form_fields() {

        $this->start_controls_section( 'section_content_form_fields', [
            'label' => esc_html__('Form fields', 'reformer-elementor'),
            'tab' => Controls_Manager::TAB_CONTENT
        ] );

        $repeater = new Repeater();

        $repeater->add_control(
            'field_type',
            [
                'label' => esc_html__( 'Field type', 'reformer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'text',
                'options' => [
                    'textarea' => esc_html__( 'Textarea', 'reformer-elementor' ),
                    'checkbox' => esc_html__( 'Checkbox', 'reformer-elementor' ),
                    'radio' => esc_html__( 'Radio button', 'reformer-elementor' ),
                    'file' => esc_html__( 'File', 'reformer-elementor' ),
                    'url' => esc_html__( 'Url', 'reformer-elementor' ),
                    'time' => esc_html__( 'Time', 'reformer-elementor' ),
                    'text' => esc_html__( 'Text', 'reformer-elementor' ),
                    'tel' => esc_html__( 'Tel', 'reformer-elementor' ),
                    'select' => esc_html__( 'Select', 'reformer-elementor' ),
                    'range' => esc_html__( 'Range', 'reformer-elementor' ),
                    'number' => esc_html__( 'Number', 'reformer-elementor' ),
                    'month' => esc_html__( 'Month', 'reformer-elementor' ),
                    'hidden' => esc_html__( 'Hidden', 'reformer-elementor' ),
                    'email' => esc_html__( 'Email', 'reformer-elementor' ),
                    'date' => esc_html__( 'Date', 'reformer-elementor' )
                ],
            ]
        );

        $repeater->add_control(
            'label_text',
            [
                'label' => esc_html__( 'Label text', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Label text', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                        'field_type!' => 'hidden'
                ]
            ]
        );

        $repeater->add_control(
            'label_position',
            [
                'label' => esc_html__( 'Label position', 'reformer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'left',
                'options' => [
                    'left' => esc_html__( 'Left', 'reformer-elementor' ),
                    'right' => esc_html__( 'Right', 'reformer-elementor' ),
                    'above' => esc_html__( 'Above', 'reformer-elementor' ),
                    'below' => esc_html__( 'Below', 'reformer-elementor' ),
                ],
                'condition' => [
                    'field_type!' => 'hidden'
                ]
            ]
        );

        $repeater->add_control(
            'allowed_file_types',
            [
                'label' => esc_html__( 'Allowed file types', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'File types', 'reformer-elementor' ),
                'default' => esc_html__( '*', 'reformer-elementor' ),
                'description' => 'Enter file types separated with coma and place dot before file extension. 
                    For example ".jpeg, .png, .gif". Input "*" symbol to accept all file types.',
                'condition' => [
                        'field_type' => 'file'
                ]
            ]
        );

        $repeater->add_control(
            'max_file_size',
            [
                'label' => esc_html__( 'Max file size', 'reformer-elementor' ),
                'type' => Controls_Manager::SELECT,
                'default' => '1',
                'options' => [
                    '1' => esc_html__( '1MB', 'reformer-elementor' ),
                    '2' => esc_html__( '2MB', 'reformer-elementor' ),
                ],
                'condition' => [
                    'field_type' => 'file'
                ]
            ]
        );

        $repeater->add_control(
            'select_options',
            [
                'label' => esc_html__( 'Options', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 10,
                'condition' => [ 'field_type' => 'select' ],
                'description' => 'Enter each option in separate line. Separate option text and value with "|" symbol. 
                    For example text|value'
            ]
        );

        $repeater->add_control(
            'multiple_select',
            [
                'label' => esc_html__( 'Multiple', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [ 'field_type' => 'select' ]
            ]
        );

        $repeater->add_control(
            'multiple_size',
            [
                'label' => esc_html__( 'Size', 'reformer-elementor' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 2,
                'max' => 100,
                'step' => 1,
                'default' => 2,
                'condition' => [
                    'field_type' => 'select',
                    'multiple_select' => 'yes'
                ]
            ]
        );


        $repeater->add_control(
            'field_name',
            [
                'label' => esc_html__( 'Name', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Name', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'field_id',
            [
                'label' => esc_html__( 'Field id', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Type field id', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'number_min',
            [
                'label' => esc_html__( 'Min', 'reformer-elementor' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 0,
                'max' => 100,
                'step' => 1,
                'default' => 0,
                'conditions' => [
                    'relation' => 'or',
                    'terms' => [
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'range'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'number'
                        ]
                    ]
                ],
            ]
        );

        $repeater->add_control(
            'number_max',
            [
                'label' => esc_html__( 'Max', 'reformer-elementor' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 100,
                'step' => 1,
                'default' => 10,
                'conditions' => [
                    'relation' => 'or',
                    'terms' => [
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'range'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'number'
                        ]
                    ]
                ],
            ]
        );


        $repeater->add_control(
            'date_min',
            [
                'label' => esc_html__( 'Date min', 'reformer-elementor' ),
                'picker_options' => [ 'enableTime' => false ],
                'type' => Controls_Manager::DATE_TIME,
                'condition' => [ 'field_type' => 'date' ]
            ]
        );

        $repeater->add_control(
            'date_max',
            [
                'label' => esc_html__( 'Date max', 'reformer-elementor' ),
                'picker_options' => [ 'enableTime' => false ],
                'type' => Controls_Manager::DATE_TIME,
                'condition' => [ 'field_type' => 'date' ]
            ]
        );

        $repeater->add_control(
            'placeholder',
            [
                'label' => esc_html__( 'Placeholder', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'placeholder', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
                'conditions' => [
                    'relation' => 'or',
                    'terms' => [
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'text'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'textarea'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'tel'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'email'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'url'
                        ],
                    ]
                ],
            ]
        );

        $repeater->add_control(
            'default_value_text',
            [
                'label' => esc_html__( 'Default value', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Default value', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
                'conditions' => [
                    'relation' => 'or',
                    'terms' => [
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'text'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'textarea'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'tel'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'email'
                        ],
                    ]
                ],
            ]
        );

        $repeater->add_control(
            'default_number',
            [
                'label' => esc_html__( 'Default value', 'reformer-elementor' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 0,
                'max' => 100,
                'step' => 1,
                'default' => 0,
                'conditions' => [
                    'relation' => 'or',
                    'terms' => [
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'range'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'number'
                        ]
                    ]
                ],
            ]
        );

        $repeater->add_control(
            'pattern',
            [
                'label' => esc_html__( 'Pattern', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'description' => esc_html__(
                    'Pattern example [0-9]{3}-[0-9]{2}-[0-9]{3}',
                  'reformer-elementor'
                ),
                'condition' => [
                     'field_type' => 'tel'
                ],
            ]
        );


        $repeater->add_control(
            'field_step',
            [
                'label' => esc_html__( 'Step', 'reformer-elementor' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 100,
                'step' => 1,
                'default' => 1,
                'condition' => [
                    'field_type' => 'number'
                ],
            ]
        );

        $repeater->add_control(
            'default_date',
            [
                'label' => esc_html__( 'Default value', 'reformer-elementor' ),
                'picker_options' => [ 'enableTime' => false ],
                'type' => Controls_Manager::DATE_TIME,
                'condition' => [
                    'field_type' => 'date',
                    'current_date_default!' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'current_date_default',
            [
                'label' => esc_html__( 'Current date by default', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'field_type' => 'date'
                ]
            ]
        );

        $repeater->add_control(
            'current_month_default',
            [
                'label' => esc_html__( 'Current month by default', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'field_type' => 'month'
                ]
            ]
        );

        $repeater->add_control(
            'default_time',
            [
                'label' => esc_html__( 'Default time', 'reformer-elementor' ),
                'picker_options' => [
                        'enableTime' => true,
                        'noCalendar' => true
                ],
                'type' => Controls_Manager::DATE_TIME,
                'condition' => [ 'field_type' => 'time' ]
            ]
        );

        $repeater->add_control(
            'default_url',
            [
                'label' => esc_html__( 'Default value', 'reformer-elementor' ),
                'type' => Controls_Manager::URL,
                'placeholder' => esc_html__( 'https://your-link.com', 'reformer-elementor' ),
                'show_external' => true,
                'default' => [
                    'url' => '',
                    'is_external' => true,
                    'nofollow' => true,
                ],
                'condition' => [ 'field_type' => 'url' ]
            ]
        );

        $repeater->add_control(
            'field_value',
            [
                'label' => esc_html__( 'Value', 'reformer-elementor' ),
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Value', 'reformer-elementor' ),
                'dynamic' => [
                    'active' => true,
                ],
                'conditions' => [
                    'relation' => 'or',
                    'terms' => [
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'checkbox'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'hidden'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'radio'
                        ]
                    ]
                ],
            ]
        );

        $repeater->add_control(
            'checked',
            [
                'label' => esc_html__( 'Checked', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
                'conditions' => [
                    'relation' => 'or',
                    'terms' => [
                        [
                           'name' => 'field_type',
                           'operator' => '===',
                           'value' => 'checkbox'
                        ],
                        [
                            'name' => 'field_type',
                            'operator' => '===',
                            'value' => 'radio'
                        ]
                    ]
                ],
            ]
        );

        $repeater->add_control(
            'required',
            [
                'label' => esc_html__( 'Required', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                        'field_type!' => 'hidden'
                ]
            ]
        );


        $repeater->add_control(
            'input_height',
            [
                'label' => esc_html__( 'Height', 'reformer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} input, {{WRAPPER}} {{CURRENT_ITEM}} select, {{WRAPPER}} {{CURRENT_ITEM}} textarea' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $repeater->add_control(
            'input_width',
            [
                'label' => esc_html__( 'Width', 'reformer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}}' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'form_fields_list',
            [
                'label' => esc_html__( 'Form fields', 'reformer-elementor' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'label_text' => 'Name',
                        'field_type' => 'text'
                    ],
                    [
                        'label_text' => 'Email',
                        'field_type' => 'email'
                    ],
                    [
                        'label_text' => 'Description',
                        'field_type' => 'textarea'
                    ]
                ],
                'title_field' => '{{{ field_type }}}',
            ]
        );

        $this->add_control(
            'required_mark',
            [
                'label' => esc_html__( 'Required field mark', 'reformer-elementor' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'reformer-elementor' ),
                'label_off' => esc_html__( 'No', 'reformer-elementor' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

    }

    /**
     * Function for generating margin padding controls.
     *
     * @param $section_id
     * @param $html_class
     * @return void
     * @since 1.0.0
     * @access private
     */
    private function generate_margin_padding_controls( $section_id, $html_class ) {
        $this->add_responsive_control(
            $section_id.'_margin',
            [
                'label' => esc_html__( 'Margin', 'reformer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    "{{WRAPPER}} .$html_class" => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
                ],
            ]
        );

        $this->add_responsive_control(
            $section_id.'_padding',
            [
                'label' => esc_html__( 'Padding', 'reformer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'separator' => 'after',
                'selectors' => [
                    "{{WRAPPER}} .$html_class" => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
                ],
            ]
        );
    }

    /**
     * Function for generating typography and tabs controls.
     *
     * @param $section_id
     * @param $opts
     * @return void
     * @since 1.0.0
     * @access private
     */
    private function generate_typography_tabs_controls( $section_id, $opts = [] ) {
        $style_opts = [
            'html_class' => array_key_exists( 'html_class', $opts ) ?
                $opts['html_class'] : '',
            'active_class' => array_key_exists( 'active_class', $opts ) ?
                $opts['active_class'] : '',
            'include_color' => array_key_exists( 'include_color', $opts ) ?
                $opts['include_color'] : true,
            'include_border' => array_key_exists( 'include_border', $opts ) ?
                $opts['include_border'] : true,
            'include_bg' => array_key_exists( 'include_bg', $opts ) ?
                $opts['include_color'] : true,
            'include_typography' => array_key_exists( 'include_typography', $opts ) ?
                $opts['include_typography'] : true,
            'include_transition' => array_key_exists( 'include_transition', $opts ) ?
                $opts['include_transition'] : true,
            'additional_color' => array_key_exists( 'additional_color', $opts ) ?
                $opts['additional_color'] : false,
            'additional_background' => array_key_exists( 'additional_background', $opts ) ?
                $opts['additional_background'] : false,
            'include_outline' => array_key_exists( 'include_outline', $opts ) ?
                $opts['include_outline'] : false,
            'additional_background_name' => array_key_exists( 'additional_background_name', $opts ) ?
                $opts['additional_background_name'] : '',
            'additional_background_class' => array_key_exists( 'additional_background_class', $opts ) ?
                $opts['additional_background_class'] : '',
            'additional_border_radius_class' => array_key_exists( 'additional_border_radius_class', $opts ) ?
                $opts['additional_border_radius_class'] : '',
            'additional_border_radius_hover_class' => array_key_exists( 'additional_border_radius_hover_class', $opts ) ?
                $opts['additional_border_radius_hover_class'] : '',
            'include_active_tab' => array_key_exists( 'include_active_tab', $opts ) ?
                $opts['include_active_tab'] : false,
            'active_tab_name' => array_key_exists( 'active_tab_name', $opts ) ?
                $opts['active_tab_name'] : 'FOCUS',
            'color_prefix' => array_key_exists( 'color_prefix', $opts ) ?
                $opts['color_prefix'] : '',
            'color_class' => array_key_exists( 'color_class', $opts ) ?
                $opts['color_class'] : '',
            'color_hover_class' => array_key_exists( 'color_hover_class', $opts ) ?
                $opts['color_hover_class'] : '',
            'color_active_class' => array_key_exists( 'color_active_class', $opts ) ?
                $opts['color_active_class'] : '',
            'color_hover_selector' => array_key_exists( 'color_hover_selector', $opts ) ?
                $opts['color_hover_selector'] : '',
            'additional_color_name' => array_key_exists( 'additional_color_name', $opts ) ?
                $opts['additional_color_name'] : '',
            'additional_color_class' => array_key_exists( 'additional_color_class', $opts ) ?
                $opts['additional_color_class'] : '',
            'additional_color_hover_class' => array_key_exists( 'additional_color_hover_class', $opts ) ?
                $opts['additional_color_hover_class'] : '',
            'additional_color_active_class' => array_key_exists( 'additional_color_active_class', $opts ) ?
                $opts['additional_color_active_class'] : '',
            'additional_transition_selector' => array_key_exists( 'additional_transition_selector', $opts ) ?
                $opts['additional_transition_selector'] : '',
            'typography_class' => array_key_exists( 'typography_class', $opts ) ?
                $opts['typography_class'] : '',
            'color_scheme_default' => array_key_exists( 'color_scheme_default', $opts ) ?
                $opts['color_scheme_default'] : Color::COLOR_3,
            'additional_color_scheme_default' => array_key_exists( 'additional_color_scheme_default', $opts ) ?
                $opts['additional_color_scheme_default'] : Color::COLOR_3
        ];


        if ( $style_opts['include_typography'] ) {
            $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => $section_id . '_typography',
                    'label' => esc_html__('Typography', 'reformer-elementor'),
                    'scheme' => Typography::TYPOGRAPHY_1,
                    'selector' => "{{WRAPPER}} .".$style_opts['typography_class'],
                ]
            );
        }

        $this->start_controls_tabs( $section_id.'_style_tabs' );

        $this->start_controls_tab(
                $section_id.'_normal_style_tab',
                ['label' => esc_html__( 'NORMAL', 'reformer-elementor' )]
        );

        if ( $style_opts['include_color'] ) {
            $this->add_control(
                $section_id . '_normal_text_color',
                [
                    'label' => esc_html__($style_opts['color_prefix'].'Color', 'reformer-elementor'),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['color_class'] => 'color: {{VALUE}} !important;',
                    ],
                ]
            );

        }

        if ( $style_opts['additional_color'] ) {
            $this->add_control(
                $section_id . '_' . $style_opts['additional_color_name'] . '_normal_text_color',
                [
                    'label' => esc_html__( $style_opts['additional_color_name'], 'reformer-elementor' ),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['additional_color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['additional_color_class'] => 'color: {{VALUE}} !important;',

                    ],
                ]
            );
        }

        if ( $style_opts['include_bg'] ) {

            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => $section_id . '_normal_background',
                    'label' => esc_html__('Background type', 'reformer-elementor'),
                    'types' => ['classic', 'gradient', 'video'],
                    'selector' => "{{WRAPPER}} .".$style_opts['html_class'],
                ]
            );

        }

        if ( $style_opts['additional_background'] ) {

            $this->add_control(
                $section_id . '_' . $style_opts['additional_background_name'] . '_normal_text_color',
                [
                    'label' => esc_html__( $style_opts['additional_background_name'], 'reformer-elementor' ),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['additional_color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['additional_background_class'] => 'background: {{VALUE}} !important;',

                    ],
                ]
            );

        }

        $this->add_control(
            $section_id . '_separate_normal',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        if ( $style_opts['include_border'] ) {

            $this->add_group_control(
                Group_Control_Border::get_type(),
                [
                    'name' => $section_id . '_border_normal',
                    'label' => esc_html__('Border Type', 'reformer-elementor'),
                    'selector' => "{{WRAPPER}} ." . $style_opts['html_class'],
                ]
            );

        }

        $this->add_responsive_control(
            $section_id.'_border_radius_normal',
            [
                'label' => esc_html__( 'Border radius', 'reformer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    "{{WRAPPER}} .".$style_opts['html_class'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    $style_opts['additional_border_radius_class'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => $section_id.'_box_shadow_normal',
                'label' => esc_html__( 'Box Shadow', 'reformer-elementor' ),
                'selector' => "{{WRAPPER}} .".$style_opts['html_class'],
            ]
        );


        $this->end_controls_tab();

        $this->start_controls_tab(
                $section_id.'_hover_style_tab',
                      ['label' => esc_html__( 'HOVER', 'reformer-elementor' )]
        );

        if ( $style_opts['include_color'] ) {
            $this->add_control(
                $section_id . '_hover_color',
                [
                    'label' => esc_html__($style_opts['color_prefix'].'Color', 'reformer-elementor'),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['color_hover_class'] => 'color: {{VALUE}} !important;',
                    ],
                ]
            );
        }

        if ( $style_opts['additional_color'] ) {
            $this->add_control(
                $section_id . '_' . $style_opts['additional_color_name'] . '_hover_text_color',
                [
                    'label' => esc_html__( $style_opts['additional_color_name'], 'reformer-elementor'),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['additional_color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['additional_color_hover_class'] => 'color: {{VALUE}} !important;',
                    ],
                ]
            );
        }

        if ( $style_opts['include_bg'] ) {
            $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => $section_id . '_background_hover',
                    'label' => esc_html__('Background type', 'reformer-elementor'),
                    'types' => ['classic', 'gradient', 'video'],
                    'selector' => "{{WRAPPER}} .".$style_opts['html_class'].":hover",
                ]
            );
        }

        if ( $style_opts['additional_background'] ) {

            $this->add_control(
                $section_id . '_' . $style_opts['additional_background_name'] . '_hover_text_color',
                [
                    'label' => esc_html__( $style_opts['additional_background_name'], 'reformer-elementor' ),
                    'type' => Controls_Manager::COLOR,
                    'scheme' => [
                        'type' => Color::get_type(),
                        'value' => $style_opts['additional_color_scheme_default'],
                    ],
                    'selectors' => [
                        "{{WRAPPER}} .".$style_opts['additional_background_class'].':hover' => 'background: {{VALUE}} !important;',

                    ],
                ]
            );

        }


        if ( $style_opts['include_transition'] ) {
            $this->add_control(
                $section_id.'_hover_transition',
                [
                    'label' => esc_html__( 'Hover transition duration', 'reformer-elementor' ),
                    'type' => Controls_Manager::SLIDER,
                    'size_units' => [ 's' ],
                    'range' => [
                        's' => [
                            'min' => 0.1,
                            'max' => 5,
                            'step' => 0.1,
                        ],
                    ],
                    'default' => [
                        'unit' => 's',
                        'size' => 0,
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .'.$style_opts['html_class'] => 'transition: color {{SIZE}}{{UNIT}}, background {{SIZE}}{{UNIT}}, box-shadow {{SIZE}}{{UNIT}}, border-radius {{SIZE}}{{UNIT}}, border {{SIZE}}{{UNIT}}, filter {{SIZE}}{{UNIT}}, stroke {{SIZE}}{{UNIT}};',
                        $style_opts['additional_transition_selector'] => 'transition: color {{SIZE}}{{UNIT}}, background {{SIZE}}{{UNIT}}, box-shadow {{SIZE}}{{UNIT}}, border-radius {{SIZE}}{{UNIT}}, border {{SIZE}}{{UNIT}}, filter {{SIZE}}{{UNIT}}, stroke {{SIZE}}{{UNIT}};;'
                    ],
                ]
            );
        }

        $this->add_control(
            $section_id.'_separate_hover',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        if ( $style_opts['include_border'] ) {

            $this->add_group_control(
                Group_Control_Border::get_type(),
                [
                    'name' => $section_id . '_border_hover',
                    'label' => esc_html__('Border Type', 'reformer-elementor'),
                    'selector' => "{{WRAPPER}} ." . $style_opts['html_class'] . ":hover",
                ]
            );

        }

        $this->add_responsive_control(
            $section_id.'_border_radius_hover',
            [
                'label' => esc_html__( 'Border radius', 'reformer-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    "{{WRAPPER}} .".$style_opts['html_class'].":hover" => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    $style_opts['additional_border_radius_hover_class'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => $section_id.'_box_shadow_hover',
                'label' => esc_html__( 'Box Shadow', 'reformer-elementor' ),
                'selector' => "{{WRAPPER}} .".$style_opts['html_class'].":hover",
            ]
        );

        $this->end_controls_tab();

        if ( $style_opts['include_active_tab'] ) {

            $this->start_controls_tab(
                    $section_id . '_focus_style_tab',
                    ['label' => esc_html__( $style_opts['active_tab_name'], 'reformer-elementor' )]
            );

            if ( $style_opts['include_color'] ) {
                $this->add_control(
                    $section_id . '_active_color',
                    [
                        'label' => esc_html__( $style_opts['color_prefix'] . 'Color', 'reformer-elementor '),
                        'type' => Controls_Manager::COLOR,
                        'scheme' => [
                            'type' => Color::get_type(),
                            'value' => $style_opts['color_scheme_default'],
                        ],
                        'selectors' => [
                            "{{WRAPPER}} ." . $style_opts['color_active_class'] => 'color: {{VALUE}} !important;',
                        ],
                    ]
                );
            }

            if ( $style_opts['additional_color'] ) {
                $this->add_control(
                    $section_id . '_' . $style_opts['additional_color_name'] . '_active_text_color',
                    [
                        'label' => esc_html__( $style_opts['additional_color_name'], 'reformer-elementor' ),
                        'type' => Controls_Manager::COLOR,
                        'scheme' => [
                            'type' => Color::get_type(),
                            'value' => $style_opts['additional_color_scheme_default'],
                        ],
                        'selectors' => [
                            "{{WRAPPER}} ." . $style_opts['additional_color_active_class'] => 'color: {{VALUE}} !important;',
                        ],
                    ]
                );
            }

            if ( $style_opts['include_bg'] ) {
                $this->add_group_control(
                    Group_Control_Background::get_type(),
                    [
                        'name' => $section_id . '_background_active',
                        'label' => esc_html__( 'Background type', 'reformer-elementor' ),
                        'types' => ['classic', 'gradient', 'video'],
                        'selector' => "{{WRAPPER}} ." . $style_opts['active_class'],
                    ]
                );
            }

            if ( $style_opts['include_outline'] ) {
                $this->add_control(
                    $section_id . '_outline_style',
                    [
                        'label' => esc_html__( 'Outline', 'reformer-elementor' ),
                        'type' => Controls_Manager::SELECT,
                        'default' => '',
                        'options' => [
                            'none' => esc_html__( 'None', 'reformer-elementor' ),
                            '' => esc_html__( 'Default', 'reformer-elementor' ),
                            'auto' => esc_html__( 'Auto', 'reformer-elementor' ),
                            'solid'  => esc_html__( 'Solid', 'reformer-elementor' ),
                            'dotted' => esc_html__( 'Dotted', 'reformer-elementor' ),
                            'double' => esc_html__( 'Double', 'reformer-elementor' ),
                            'dashed' => esc_html__( 'Dashed', 'reformer-elementor' ),
                            'groove' => esc_html__( 'Groove', 'reformer-elementor' ),
                            'ridge' => esc_html__( 'Ridge', 'reformer-elementor' ),
                            'inset' => esc_html__( 'Inset', 'reformer-elementor' ),
                            'outset' => esc_html__( 'Outset', 'reformer-elementor' ),
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .' . $style_opts['html_class'] . ':focus' => 'outline-style: {{VALUE}};',
                        ],
                    ]
                );

                $this->add_control(
                    $section_id . '_outline_color',
                    [
                        'label' => esc_html__( 'Outline color', 'reformer-elementor' ),
                        'type' => Controls_Manager::COLOR,
                        'condition' => [
                            $section_id . '_outline_style!' => 'none'
                        ],
                        'selectors' => [
                            "{{WRAPPER}} ." . $style_opts['html_class'] . ':focus' => 'outline-color: {{VALUE}} !important;',
                        ],
                    ]
                );

                $this->add_control(
                    $section_id . '_custom_outline_width',
                    [
                        'label' => esc_html__( 'Custom outline width', 'reformer-elementor' ),
                        'type' => Controls_Manager::SWITCHER,
                        'return_value' => 'yes',
                        'default' => 'no',
                        'condition' => [
                            $section_id . '_outline_style!' => 'none'
                        ],
                    ]
                );

                $this->add_control(
                    $section_id . '_outline_width',
                    [
                        'label' => esc_html__( 'Outline width', 'reformer-elementor' ),
                        'type' => Controls_Manager::SELECT,
                        'default' => '',
                        'options' => [
                            '' => esc_html__( 'Default', 'reformer-elementor' ),
                            'thin' => esc_html__( 'Thin', 'reformer-elementor' ),
                            'medium'  => esc_html__( 'Medium', 'reformer-elementor' ),
                            'thick' => esc_html__( 'Thick', 'reformer-elementor' ),
                        ],
                        'condition' => [
                            $section_id . '_custom_outline_width!' => 'yes',
                            $section_id . '_outline_style!' => 'none'
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .' . $style_opts['html_class'] . ':focus' => 'outline-width: {{VALUE}};',
                        ],
                    ]
                );

                $this->add_responsive_control(
                    $section_id . '_outline_custom_width',
                    [
                        'label' => esc_html__( 'Outline width', 'reformer-elementor' ),
                        'type' => Controls_Manager::SLIDER,
                        'size_units' => [ 'px', 'em' ],
                        'range' => [
                            'px' => [
                                'min' => 0,
                                'max' => 500,
                                'step' => 1,
                            ],
                            'em' => [
                                'min' => 0.1,
                                'max' => 100,
                                'step' => 0.1
                            ]
                        ],
                        'condition' => [
                            $section_id . '_custom_outline_width' => 'yes',
                            $section_id . '_outline_style!' => 'none'
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .' . $style_opts['html_class'] . ':focus' => 'outline-width: {{SIZE}}{{UNIT}};',
                        ],
                    ]
                );


            }

            $this->add_control(
                $section_id . '_separate_active',
                [
                    'type' => Controls_Manager::DIVIDER,
                ]
            );

            if ( $style_opts['include_border'] ) {

                $this->add_group_control(
                    Group_Control_Border::get_type(),
                    [
                        'name' => $section_id . '_border_active',
                        'label' => esc_html__('Border Type', 'reformer-elementor'),
                        'selector' => "{{WRAPPER}} ." . $style_opts['active_class'],
                    ]
                );

            }

            $this->add_responsive_control(
                $section_id . '_border_radius_active',
                [
                    'label' => esc_html__( 'Border radius', 'reformer-elementor' ),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%', 'em'],
                    'selectors' => [
                        "{{WRAPPER}} ." . $style_opts['active_class'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
            );

            $this->add_group_control(
                Group_Control_Box_Shadow::get_type(),
                [
                    'name' => $section_id . '_box_shadow_active',
                    'label' => esc_html__( 'Box Shadow', 'reformer-elementor' ),
                    'selector' => "{{WRAPPER}} ." . $style_opts['active_class'],
                ]
            );

            $this->end_controls_tab();

        }


        $this->end_controls_tabs();
    }

    /**
     * Add widget controls: Style -> Section Style Title.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_title() {

        $this->start_controls_section( 'section_style_title', [
            'label' => esc_html__( 'Title', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE,
            'condition' => [ 'show_title' => 'yes' ]
        ] );

        $this->generate_margin_padding_controls(
        'section_style_title',
        'mdp-reformer-elementor-form-title'
        );

        $this->generate_typography_tabs_controls( 'section_style_title', [
            'html_class' => 'mdp-reformer-elementor-form-title',
            'color_class' => 'mdp-reformer-elementor-form-title',
            'color_hover_class' => 'mdp-reformer-elementor-form-title:hover',
            'typography_class' => 'mdp-reformer-elementor-form-title'
        ] );


        $this->end_controls_section();

    }

    /**
     * Add widget controls: Style -> Section Style Description.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_description() {

        $this->start_controls_section( 'section_style_description', [
            'label' => esc_html__( 'Description', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE,
            'condition' => [ 'show_description' => 'yes' ]
        ] );

        $this->generate_margin_padding_controls(
        'section_style_description',
        'mdp-reformer-elementor-form-description'
        );

        $this->generate_typography_tabs_controls( 'section_style_description', [
            'html_class' => 'mdp-reformer-elementor-form-description',
            'color_class' => 'mdp-reformer-elementor-form-description',
            'color_hover_class' => 'mdp-reformer-elementor-form-description:hover',
            'typography_class' => 'mdp-reformer-elementor-form-description'
        ] );

        $this->end_controls_section();

    }

    /**
     * Add widget controls: Style -> Section Style Field Label.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_label()
    {

        $this->start_controls_section('section_style_label', [
            'label' => esc_html__( 'Label', 'reformer-elementor' ),
            'tab' => Controls_Manager::TAB_STYLE
        ]);

        $this->generate_margin_padding_controls(
        'section_style_label',
        'mdp-reformer-elementor-form .mdp-reformer-elementor-field-label'
        );

        $this->add_responsive_control(
            'label_min_width',
            [
                'label' => esc_html__( 'Min width', 'reformer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-reformer-elementor-form .mdp-reformer-elementor-field-label' => 'min-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );



        $this->generate_typography_tabs_controls( 'section_style_label', [
            'html_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-field-label',
            'additional_color' => true,
            'additional_color_name' => 'Required mark color',
            'additional_color_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-field-required-mark::after',
            'additional_color_hover_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-field-required-mark:hover::after',
            'color_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-field-label',
            'color_hover_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-field-label:hover',
            'typography_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-field-label',
            'additional_transition_selector' => '{{WRAPPER}} .mdp-reformer-elementor-field-label::after'
        ] );


        $this->end_controls_section();
    }


        /**
     * Add widget controls: Style -> Section Style Input.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_input() {

        $this->start_controls_section( 'section_style_input', [
            'label' => esc_html__( 'Input', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls(
            'section_style_input',
            'mdp-reformer-elementor-form .mdp-reformer-elementor-input-field'
        );

        $this->generate_typography_tabs_controls( 'section_style_input', [
            'html_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-input-field',
            'active_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-input-field:focus',
            'typography_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-input-field, {{WRAPPER}} .mdp-reformer-elementor-form .mdp-reformer-elementor-input-field:hover::placeholder',
            'include_active_tab' => true,
            'additional_color' => true,
            'include_outline' => true,
            'additional_color_name' => 'Placeholder color',
            'color_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-input-field',
            'color_hover_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-input-field:hover',
            'color_active_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-input-field:focus',
            'additional_color_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-input-field::placeholder',
            'additional_color_hover_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-input-field:hover::placeholder'
        ] );

        $this->end_controls_section();

    }

    /**
     * Add widget controls: Style -> Section Style Textarea.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_textarea() {

        $this->start_controls_section( 'section_style_textarea', [
            'label' => esc_html__( 'Textarea', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls(
        'section_style_textarea',
        'mdp-reformer-elementor-form textarea'
        );

        $this->generate_typography_tabs_controls( 'section_style_textarea', [
            'html_class' => 'mdp-reformer-elementor-form textarea',
            'active_class' => 'mdp-reformer-elementor-form textarea:focus',
            'typography_class' => 'mdp-reformer-elementor-form textarea, {{WRAPPER}} .mdp-reformer-elementor-form textarea:hover::placeholder',
            'include_active_tab' => true,
            'additional_color' => true,
            'include_outline' => true,
            'additional_color_name' => 'Placeholder color',
            'color_class' => 'mdp-reformer-elementor-form textarea',
            'color_hover_class' => 'mdp-reformer-elementor-form textarea:hover',
            'color_active_class' => 'mdp-reformer-elementor-form textarea:focus',
            'additional_color_class' => 'mdp-reformer-elementor-form textarea::placeholder',
            'additional_color_hover_class' => 'mdp-reformer-elementor-form textarea:hover::placeholder'
        ] );

        $this->end_controls_section();

    }

    /**
     * Add widget controls: Style -> Section Style Range.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    public function section_style_range() {
        $this->start_controls_section( 'section_style_range', [
            'label' => esc_html__( 'Range', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls(
            'section_style_range',
            'mdp-reformer-elementor-range-field, {{WRAPPER}} .mdp-reformer-elementor-range-output'
        );

        $this->add_control(
            'thumb_height',
            [
                'label' => esc_html__( 'Thumb height', 'reformer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    'default' => [
                        'unit' => 'px',
                        'size' => 40,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-reformer-elementor-range-field::-webkit-slider-thumb' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'thumb_width',
            [
                'label' => esc_html__( 'Thumb width', 'reformer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    'default' => [
                        'unit' => 'px',
                        'size' => 20,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-reformer-elementor-range-field::-webkit-slider-thumb' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->generate_typography_tabs_controls( 'section_style_range', [
            'html_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-range-field',
            'additional_background' => true,
            'additional_background_name' => 'Thumb color',
            'additional_background_class' => 'mdp-reformer-elementor-range-field::-webkit-slider-thumb',
            'additional_background_hover_class' => 'mdp-reformer-elementor-range-field:hover::-webkit-slider-thumb',
            'additional_border_radius_hover_class' => '{{WRAPPER}} .mdp-reformer-elementor-range-field:hover::-webkit-slider-thumb',
            'additional_border_radius_class' => '{{WRAPPER}} .mdp-reformer-elementor-range-field::-webkit-slider-thumb',
            'typography_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-range-output',
            'additional_transition_selector' => '{{WRAPPER}} .mdp-reformer-elementor-range-field::-webkit-slider-thumb',
            'active_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-range-field:focus',
            'include_active_tab' => true,
            'include_outline' => true,
            'color_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-range-output',
            'color_hover_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-range-field-wrapper:hover .mdp-reformer-elementor-range-output',
            'color_active_class' => 'mdp-reformer-elementor-form .mdp-reformer-elementor-range-field:focus + .mdp-reformer-elementor-range-output',
        ] );

        $this->end_controls_section();
    }


    /**
     * Add widget controls: Style -> Section Style Radio Button.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_radio_button() {

        $this->start_controls_section( 'section_style_radio_button', [
            'label' => esc_html__( 'Radio button', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls(
        'section_style_radio_button',
        'mdp-reformer-elementor-form input[type="radio"]'
        );

        $this->add_control(
            'radio_button_size',
            [
                'label' => esc_html__( 'Scale', 'reformer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0.1,
                        'max' => 10,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-reformer-elementor-form input[type="radio"]' => 'transform: scale({{SIZE}})',
                ],
            ]
        );

        $this->add_control(
            'radio_button_width_divider',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        $this->generate_typography_tabs_controls( 'section_style_radio_button', [
            'html_class' => 'mdp-reformer-elementor-form input[type="radio"]',
            'include_color' => false,
            'include_border' => false,
            'active_tab_name' => 'CHECKED',
            'include_active_tab' => true,
            'include_outline' => true,
            'active_class' => 'mdp-reformer-elementor-form input[type="radio"]:checked'
        ] );


        $this->end_controls_section();

    }

    /**
     * Add widget controls: Style -> Section Style Checkbox.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_checkbox() {

        $this->start_controls_section( 'section_style_checkbox', [
            'label' => esc_html__( 'Checkbox', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls(
        'section_style_checkbox',
        'mdp-reformer-elementor-form input[type="checkbox"]'
        );

        $this->add_control(
            'checkbox_size',
            [
                'label' => esc_html__( 'Scale', 'reformer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0.1,
                        'max' => 10,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-reformer-elementor-form input[type="checkbox"]' => 'transform: scale({{SIZE}})',
                ],
            ]
        );

        $this->add_control(
            'checkbox_divider',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        $this->generate_typography_tabs_controls( 'section_style_checkbox', [
            'html_class' => 'mdp-reformer-elementor-form input[type="checkbox"]',
            'include_color' => false,
            'include_border' => false,
            'active_tab_name' => 'CHECKED',
            'include_active_tab' => true,
            'include_outline' => true,
            'active_class' => 'mdp-reformer-elementor-form input[type="checkbox"]:checked'
        ] );


        $this->end_controls_section();

    }

    /**
     * Add widget controls: Style -> Section Style Select.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_select() {

        $this->start_controls_section( 'section_style_select', [
            'label' => esc_html__( 'Select', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls(
        'section_style_select',
        'mdp-reformer-elementor-form select'
        );

        $this->generate_typography_tabs_controls( 'section_style_select', [
            'html_class' => 'mdp-reformer-elementor-form select',
            'typography_class' => 'mdp-reformer-elementor-form select',
            'color_class' => 'mdp-reformer-elementor-form select option',
            'color_hover_class' => 'mdp-reformer-elementor-form select:hover',
            'additional_transition_selector' => '{{WRAPPER}} .mdp-reformer-elementor-form select option'
        ] );


        $this->end_controls_section();

    }


    /**
     * Add widget controls: Style -> Section Style Button.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_button() {

        $this->start_controls_section( 'section_style_button', [
            'label' => esc_html__( 'Button', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls(
        'section_style_button',
        'mdp-reformer-elementor-form input[type="submit"]'
        );

        $this->add_control(
            'button_width',
            [
                'label' => esc_html__( 'Width', 'reformer-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-reformer-elementor-form input[type="submit"]' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_width_separate',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        $this->generate_typography_tabs_controls( 'section_style_button', [
            'html_class' => 'mdp-reformer-elementor-form input[type="submit"]',
            'typography_class' => 'mdp-reformer-elementor-form input[type="submit"]',
            'color_class' => 'mdp-reformer-elementor-form input[type="submit"]',
            'color_hover_class' => 'mdp-reformer-elementor-form input[type="submit"]:hover'
        ] );


        $this->end_controls_section();

    }


    /**
     * Add widget controls: Style -> Section Style Success Message.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_success() {

        $this->start_controls_section( 'section_style_success', [
            'label' => esc_html__( 'Success message', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls(
        'section_style_success',
        'mdp-reformer-elementor-success-message'
        );

        $this->generate_typography_tabs_controls( 'section_style_success', [
            'html_class' => 'mdp-reformer-elementor-success-message',
            'typography_class' => 'mdp-reformer-elementor-success-message',
            'color_class' => 'mdp-reformer-elementor-success-message',
            'color_hover_class' => 'mdp-reformer-elementor-success-message:hover',
        ] );


        $this->end_controls_section();

    }

    /**
     * Add widget controls: Style -> Section Style Error Message.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function section_style_errors() {

        $this->start_controls_section( 'section_style_errors', [
            'label' => esc_html__( 'Error message', 'reformer-elementor' ),
            'tab'   => Controls_Manager::TAB_STYLE
        ] );

        $this->generate_margin_padding_controls(
        'section_style_errors',
        'mdp-reformer-elementor-error-message'
        );

        $this->generate_typography_tabs_controls( 'section_style_errors', [
            'html_class' => 'mdp-reformer-elementor-error-message',
            'typography_class' => 'mdp-reformer-elementor-error-message',
            'color_class' => 'mdp-reformer-elementor-error-message',
            'color_hover_class' => 'mdp-reformer-elementor-error-message:hover',
        ] );


        $this->end_controls_section();

    }

    /**
     * Get form title method.
     *
     * @param $settings
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function get_form_title( $settings ) {
        if ( $settings['show_title'] === 'yes' ) {
            return sprintf(
            '<div class="mdp-reformer-elementor-form-title">%s</div>',
                esc_html( $settings['form_title'] ) );
        }
        return '';
    }

    private function get_form_description( $settings ) {
        if ( $settings['show_description'] === 'yes' ) {
            return sprintf(
            '<div class="mdp-reformer-elementor-form-description">%s</div>',
                wp_kses_post( $settings['form_description'] ) );
        }
        return '';
    }

    /**
     * Get attributes for form fields.
     *
     * @param $field
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function get_fields_attributes( $field ) {
        $special_attributes = [];

        switch ( $field['field_type'] ) {
            case 'hidden':
            case 'radio':
            case 'checkbox':
                $special_attributes['value'] = !empty( $field['field_value'] ) ? $field['field_value'] : '';
                break;
            case 'text':
            case 'email':
                $special_attributes['placeholder'] = !empty( $field['placeholder'] ) ? $field['placeholder'] : '';
                $special_attributes['value'] = !empty( $field['default_value_text'] ) ?
                    $field['default_value_text'] : '';
                break;
            case 'number':
            case 'range':
                $special_attributes['min'] = !empty( $field['number_min'] ) ? $field['number_min'] : '';
                $special_attributes['max'] = !empty( $field['number_max'] ) ? $field['number_max'] : '';
                $special_attributes['step'] = !empty( $field['field_step'] ) ? $field['field_step'] : '';
                $special_attributes['value'] = !empty( $field['default_number'] ) ? $field['default_number'] : '';
                break;
            case 'month':
                $special_attributes['value'] = $field['current_month_default'] === 'yes' ?
                                               date( 'Y-m' ) :
                                               '';
                $special_attributes['placeholder'] = esc_html( 'Y-m' );
                break;
            case 'date':
                $current_date = date( 'Y-m-d' );
                $special_attributes['value'] = ( !empty( $field['default_date'] ) ) ?
                                               $field['default_date'] :
                                               ( ( $field['current_date_default'] === 'yes' ) ?
                                                $current_date :
                                               '' );
                $special_attributes['placeholder'] = esc_html( 'mm/dd/yyyy' );
                break;
            case 'tel':
                $special_attributes['value'] = !empty( $field['default_value_text'] ) ?
                    $field['default_value_text'] : '';
                $special_attributes['pattern'] = !empty( $field['pattern'] ) ? $field['pattern'] : '';
                $special_attributes['placeholder'] = !empty( $field['placeholder'] ) ?
                    $field['placeholder'] : '';
                break;
            case 'url':
                $special_attributes['value'] = !empty( $field['default_url']['url'] ) ?
                    $field['default_url']['url'] : '';
                $special_attributes['placeholder'] = !empty( $field['placeholder'] ) ?
                    $field['placeholder'] : '';
                break;
            case 'time':
                $special_attributes['value'] = !empty( $field['default_time'] ) ? $field['default_time'] : '';
                break;
        }

        $field_attributes_str = '';

        foreach ( $special_attributes as $name => $attribute ) {
            if ( !empty( $attribute ) ) {
                $field_attributes_str .= ''.$name.' = "'.$attribute.'" ';
            }
        }
        return $field_attributes_str;
    }

    /**
     * Generate hash.
     *
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function generate_app_hash() {
        return substr( hash( 'ripemd160', date('l jS \of F Y h:i:s A') ),
                rand( 0 , 20 ), 3 ) . rand( 11 , 99 );
    }

    /**
     * Create fields labels method.
     *
     * @param $field
     * @param $required_mark
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function generate_label( $field, $required_mark ) {
        if ( !empty( $field['label_text'] ) && $field['field_type'] !== 'hidden' ) {
            $required_mark_class = $required_mark === 'yes' && $field['required'] === 'yes' ?
                                   'mdp-reformer-elementor-field-required-mark' :
                                   '';
            return sprintf(
             '<label for="%s" class="mdp-reformer-elementor-field-label %s">%s</label>',
                    esc_attr( $field['field_id'] ),
                    esc_attr( $required_mark_class ),
                    esc_attr( $field['label_text'] )
            );
        }
        return '';
    }

    /**
     * Create input fields method.
     *
     * @param $field
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function generate_input_field( $field ) {
        $field_name = !empty( $field['field_name'] ) ? $field['field_name'] : 'field_' . $this->generate_app_hash();
        $required = $field['required'] === 'yes' ? 'required' : '';
        $attributes = $this->get_fields_attributes( $field );
        $checked = $field['checked'] === 'yes' ? 'checked' : '';
        $html_class =  ( $field['field_type'] === 'radio' ) ?
                       'mdp-reformer-elementor-radio-field' :
                       ( ( $field['field_type'] === 'checkbox' ) ?
                       'mdp-reformer-elementor-checkbox-field' :
                       'mdp-reformer-elementor-input-field' );

        return sprintf(
            '<input class="%s" id="%s" type="%s" name="%s" %s %s %s>',
                esc_attr( $html_class ),
                esc_attr( $field['field_id'] ),
                esc_attr( $field['field_type'] ),
                esc_attr( $field_name ),
                esc_attr( $required ),
                esc_attr( $checked ),
                wp_kses_post( $attributes )
        );
    }


    /**
     * Create range field method.
     *
     * @param $field
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function generate_range_field( $field ) {
        $field_name = !empty( $field['field_name'] ) ? $field['field_name'] : 'field_' . $this->generate_app_hash();
        $required = $field['required'] === 'yes' ? 'required' : '';
        $attributes = $this->get_fields_attributes( $field );

        return sprintf(
            '<div class="mdp-reformer-elementor-range-field-wrapper">
                    <input 
                        class="mdp-reformer-elementor-range-field" 
                        oninput="this.nextElementSibling.innerHTML = this.value" 
                        id="%s" type="range" 
                        name="%s" %s %s> 
                        <span class="mdp-reformer-elementor-range-output" name="mdp_range_field_output"></span>
                        </div>',
                   esc_attr( $field['field_id'] ),
                   esc_attr( $field_name ),
                   esc_attr( $required ),
                   wp_kses_post( $attributes )
        );
    }

    /**
     * Create select field method.
     *
     * @param $field
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function generate_select_field( $field ) {
        $field_name = !empty( $field['field_name'] ) ? $field['field_name'] : 'field_' . $this->generate_app_hash();
        $required = $field['required'] === 'yes' ? 'required' : '';
        $multiple = $field['multiple_select'] === 'yes' ? 'multiple' : '';
        $size = $field['multiple_select'] === 'yes' ? $field['multiple_size'] : '';

        $select_options = [];
        $options = explode( "\n", $field['select_options'] );
        foreach ( $options as $option ) {
            $optionAttrs = explode( '|', $option );
            $select_options[] = sprintf(
                    '<option value="%s">%s</option>',
                        esc_attr( $optionAttrs[1] ),
                        esc_attr( $optionAttrs[0] )
            );
        }

        $options_str = implode( '', $select_options );

        return sprintf(
            '<select name="%s" size="%s" id="%s" %s %s>%s</select>',
                esc_attr( $field_name ),
                esc_attr( $size ),
                esc_attr( $field['field_id'] ),
                esc_attr( $multiple ),
                esc_attr( $required ),
                $options_str
        );
    }

    /**
     * Create textarea field method.
     *
     * @param $field
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function generate_textarea_field( $field ) {
        $field_name = !empty( $field['field_name'] ) ? $field['field_name'] : 'field_' . $this->generate_app_hash();
        $required = $field['required'] === 'yes' ? 'required' : '';
        return sprintf(
            '<textarea name="%s" id="%s" placeholder="%s" %s cols="30" rows="10">%s</textarea>',
                esc_attr( $field_name ),
                esc_attr( $field['field_id'] ),
                esc_attr( $field['placeholder'] ),
                esc_attr( $required ),
                esc_attr( $field['default_value_text'] )
            );
    }

    /**
     * Create file field method.
     *
     * @param $field
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function generate_file_field( $field ) {
        $field_name = !empty( $field['field_name'] ) ? $field['field_name'] : 'field_' . $this->generate_app_hash();
        $required = $field['required'] === 'yes' ? 'required' : '';

        return sprintf(
            '<input class="mdp-reformer-elementor-input-field" 
                        id="%s" data-maxsize="%s" accept="%s" type="file" name="%s" %s >',
            esc_attr( $field['field_id'] ),
            esc_attr( $field['max_file_size'] ),
            esc_attr( $field['allowed_file_types'] ),
            esc_attr( $field_name ),
            esc_attr( $required )
        );
    }

    /**
     * Create all form field method.
     *
     * @param $settings
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function generate_form_fields( $settings ) {
        $fields = [];

        foreach ( $settings['form_fields_list'] as $form_field ) {
            $position_class = 'mdp-reformer-elementor-label-pos-' . esc_attr( $form_field['label_position'] ) . ' ';
            $current_class = 'elementor-repeater-item-' . esc_attr( $form_field['_id'] );
            if ( $form_field['field_type'] === 'textarea' ) {
                $fields[] = '<div class="mdp-reformer-elementor-field-wrapper '. $position_class.$current_class.'">' .
                            $this->generate_label( $form_field, $settings['required_mark'] ) .
                            $this->generate_textarea_field( $form_field ) .
                            '</div>' ;
            } elseif ( $form_field['field_type'] === 'file' ) {
                $fields[] = '<div class="mdp-reformer-elementor-field-wrapper '. $position_class.$current_class.'">' .
                            $this->generate_label( $form_field, $settings['required_mark'] ) .
                            $this->generate_file_field( $form_field ) .
                            '</div>';
            } elseif ( $form_field['field_type'] === 'select' ) {
                $fields[] = '<div class="mdp-reformer-elementor-field-wrapper '.$position_class.$current_class. '">' .
                            $this->generate_label( $form_field, $settings['required_mark'] ) .
                            $this->generate_select_field( $form_field ) .
                            '</div>';
            } elseif ( $form_field['field_type'] === 'range' ) {
                $fields[] = '<div class="mdp-reformer-elementor-field-wrapper '.$position_class.$current_class.'">' .
                            $this->generate_label( $form_field, $settings['required_mark'] ) .
                            $this->generate_range_field( $form_field ) .
                            '</div>';
            } else {
                $fields[] = '<div class="mdp-reformer-elementor-field-wrapper '.$position_class.$current_class.'">' .
                            $this->generate_label( $form_field, $settings['required_mark'] ) .
                            $this->generate_input_field( $form_field ) .
                            '</div>';
            }
        }

        return implode( ' ', $fields );

    }

    /**
     * Create whole form method.
     *
     * @param $settings
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function generate_form( $settings ) {
        $form_id_input = sprintf(
                '<input type="hidden" name="mdp_form_id" value="%s">',
                    esc_attr( $this->get_id() )
        );
        $post_id_input = sprintf(
                '<input type="hidden" name="mdp_post_id" value="%s">',
                    esc_attr( get_the_ID() )
        );
        $form_name = !empty( $settings['form_name'] ) ? $settings['form_name'] : 'form_' . $this->generate_app_hash();
        $submit_form_button = sprintf(
            '<div class="mdp-reformer-elementor-submit-button-wrapper">
                        <input class="mdp-reformer-elementor-submit-button" name="submit_form" type="submit" value="%s">
                    </div>',
            esc_attr( $settings['submit_button_text'] )
        );

        $form_fields = $this->generate_form_fields( $settings );

        return sprintf(
                '<form name="%s" method="post" class="mdp-reformer-elementor-form">%s %s %s %s</form>
                <div class="mdp-reformer-elementor-success-message"></div>
                <div class="mdp-reformer-elementor-error-message"></div>',
                    esc_attr( $form_name ),
                    $form_id_input,
                    $post_id_input,
                    $form_fields,
                    $submit_form_button
        );
    }

    /**
     * Render Frontend Output. Generate the final HTML on the frontend.
     *
     * @access protected
     *
     * @return void
     **/
    protected function render() {
       $settings = $this->get_settings_for_display();

       echo sprintf(
               '<!-- Start Reformer for Elementor WordPress Plugin -->
                       <div class="mdp-reformer-elementor-box">%s %s %s</div>
                       <!-- End Reformer for Elementor WordPress Plugin -->',
                $this->get_form_title( $settings ),
                $this->get_form_description( $settings ),
                $this->generate_form( $settings )
       );

    }

    /**
     * Return link for documentation
     * Used to add stuff after widget
     *
     * @access public
     *
     * @return string
     **/
    public function get_custom_help_url() {

        return 'https://docs.merkulov.design/tag/reformer';

    }

}
